<?php
// Prevent any output before JSON response
ob_start();

// Disable error display but enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once '../config/db.php';
require_once '../vendor/autoload.php';

// Check if TCPDF is available
if (!class_exists('TCPDF')) {
    ob_end_clean();
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'TCPDF library not found. Please check if it is properly installed.']);
    exit;
}

// Define TCPDF constants if not already defined
if (!defined('PDF_PAGE_ORIENTATION')) {
    define('PDF_PAGE_ORIENTATION', 'P');
}
if (!defined('PDF_UNIT')) {
    define('PDF_UNIT', 'mm');
}
if (!defined('PDF_PAGE_FORMAT')) {
    define('PDF_PAGE_FORMAT', 'A4');
}

use TCPDF as TCPDF;

// Clear any output buffer and set JSON header
ob_end_clean();
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

$invoice_id = $_GET['id'] ?? null;

if (!$invoice_id) {
    echo json_encode(['success' => false, 'message' => 'Invoice ID is required']);
    exit;
}

try {
    // Start output buffering to catch any unexpected output
    ob_start();
    
    // Get invoice details
    $stmt = $pdo->prepare("
        SELECT i.*, c.name as customer_name, c.phone as customer_phone, c.email as customer_email, c.address as customer_address,
               u.username as created_by_name, q.quote_number as quotation_reference
        FROM invoices i
        LEFT JOIN customers c ON i.customer_id = c.id
        LEFT JOIN users u ON i.created_by = u.id
        LEFT JOIN quotations q ON i.quotation_id = q.id
        WHERE i.id = ?
    ");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$invoice) {
        echo json_encode(['success' => false, 'message' => 'Invoice not found']);
        exit;
    }

    // Get invoice items
    $stmt = $pdo->prepare("
        SELECT ii.*, p.name as product_name
        FROM invoice_items ii
        LEFT JOIN products p ON ii.product_id = p.id
        WHERE ii.invoice_id = ?
    ");
    $stmt->execute([$invoice_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get store settings
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_group IN ('general', 'store_info')");
    $stmt->execute();
    $settings = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }

    // Create PDF
    try {
        // Define TCPDF paths explicitly
        if (!defined('K_PATH_MAIN')) {
            define('K_PATH_MAIN', dirname(__FILE__) . '/../vendor/tecnickcom/tcpdf/');
        }
        if (!defined('K_PATH_FONTS')) {
            define('K_PATH_FONTS', K_PATH_MAIN . 'fonts/');
        }
        if (!defined('K_PATH_CACHE')) {
            define('K_PATH_CACHE', dirname(__FILE__) . '/../uploads/cache/');
        }
        
        // Create cache directory if it doesn't exist
        if (!is_dir(dirname(__FILE__) . '/../uploads/cache/')) {
            mkdir(dirname(__FILE__) . '/../uploads/cache/', 0777, true);
        }
        
        $pdf = new TCPDF('P', 'mm', 'A4', true, 'UTF-8', false);
        
        // Set document information
        $pdf->SetCreator($settings['store_name'] ?? 'POS System');
        $pdf->SetAuthor($settings['store_name'] ?? 'POS System');
        $pdf->SetTitle('Invoice - ' . $invoice['invoice_number']);
        $pdf->SetSubject('Invoice');

        // Remove default header/footer
        $pdf->setPrintHeader(false);
        $pdf->setPrintFooter(false);

        // Set margins
        $pdf->SetMargins(15, 15, 15);
        $pdf->SetAutoPageBreak(TRUE, 15);

        // Add a page
        $pdf->AddPage();

        // Set font
        $pdf->SetFont('helvetica', '', 10);
    } catch (Exception $e) {
        throw new Exception('Failed to create PDF object: ' . $e->getMessage());
    }

    // Store header
    $storeName = $settings['store_name'] ?? $settings['business_name'] ?? 'Your Store Name';
    $storeAddress = $settings['store_address'] ?? $settings['business_address'] ?? 'Store Address';
    $storePhone = $settings['store_phone'] ?? $settings['business_phone'] ?? 'Phone Number';
    $storeEmail = $settings['store_email'] ?? $settings['business_email'] ?? 'Email';

    $pdf->SetFont('helvetica', 'B', 16);
    $pdf->Cell(0, 10, $storeName, 0, 1, 'L');
    $pdf->SetFont('helvetica', '', 10);
    $pdf->Cell(0, 5, $storeAddress, 0, 1, 'L');
    $pdf->Cell(0, 5, 'Phone: ' . $storePhone, 0, 1, 'L');
    $pdf->Cell(0, 5, 'Email: ' . $storeEmail, 0, 1, 'L');

    $pdf->Ln(10);

    // Invoice title
    $pdf->SetFont('helvetica', 'B', 14);
    $pdf->Cell(0, 10, 'INVOICE', 0, 1, 'C');
    $pdf->Ln(5);

    // Invoice details
    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(0, 5, 'Invoice Number: ' . $invoice['invoice_number'], 0, 1, 'L');
    $pdf->Cell(0, 5, 'Issue Date: ' . date('M j, Y', strtotime($invoice['issue_date'])), 0, 1, 'L');
    $pdf->Cell(0, 5, 'Due Date: ' . date('M j, Y', strtotime($invoice['due_date'])), 0, 1, 'L');
    $pdf->Cell(0, 5, 'Status: ' . ucfirst($invoice['status']), 0, 1, 'L');

    $pdf->Ln(5);

    // Customer information
    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(0, 5, 'Bill To:', 0, 1, 'L');
    $pdf->SetFont('helvetica', '', 10);
    $customerName = $invoice['customer_name'] ?? 'Walk-in Customer';
    $pdf->Cell(0, 5, $customerName, 0, 1, 'L');
    
    if ($invoice['customer_phone']) {
        $pdf->Cell(0, 5, 'Phone: ' . $invoice['customer_phone'], 0, 1, 'L');
    }
    if ($invoice['customer_email']) {
        $pdf->Cell(0, 5, 'Email: ' . $invoice['customer_email'], 0, 1, 'L');
    }
    if ($invoice['customer_address']) {
        $pdf->Cell(0, 5, 'Address: ' . $invoice['customer_address'], 0, 1, 'L');
    }

    $pdf->Ln(10);

    // Items table
    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->SetFillColor(240, 240, 240);
    $pdf->Cell(80, 8, 'Item', 1, 0, 'L', true);
    $pdf->Cell(25, 8, 'Qty', 1, 0, 'C', true);
    $pdf->Cell(30, 8, 'Unit Price', 1, 0, 'R', true);
    $pdf->Cell(30, 8, 'Total', 1, 1, 'R', true);

    $pdf->SetFont('helvetica', '', 9);
    $subtotal = 0;
    foreach ($items as $item) {
        $pdf->Cell(80, 6, $item['product_name'], 1, 0, 'L');
        $pdf->Cell(25, 6, $item['quantity'], 1, 0, 'C');
        $pdf->Cell(30, 6, 'Ksh ' . number_format($item['unit_price'], 2), 1, 0, 'R');
        $pdf->Cell(30, 6, 'Ksh ' . number_format($item['total_price'], 2), 1, 1, 'R');
        $subtotal += $item['total_price'];
    }

    $pdf->Ln(5);

    // Totals
    $pdf->SetFont('helvetica', 'B', 10);
    $pdf->Cell(135, 6, 'Subtotal:', 0, 0, 'R');
    $pdf->Cell(30, 6, 'Ksh ' . number_format($subtotal, 2), 0, 1, 'R');

    $taxAmount = $invoice['tax_amount'] ?? 0;
    $pdf->Cell(135, 6, 'Tax (0%):', 0, 0, 'R');
    $pdf->Cell(30, 6, 'Ksh ' . number_format($taxAmount, 2), 0, 1, 'R');

    $pdf->SetFont('helvetica', 'B', 12);
    $pdf->Cell(135, 8, 'Total:', 0, 0, 'R');
    $pdf->Cell(30, 8, 'Ksh ' . number_format($invoice['total_amount'], 2), 0, 1, 'R');

    // Notes
    if ($invoice['notes']) {
        $pdf->Ln(10);
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->Cell(0, 5, 'Notes:', 0, 1, 'L');
        $pdf->SetFont('helvetica', '', 9);
        $pdf->MultiCell(0, 5, $invoice['notes'], 0, 'L');
    }

    // Generate PDF file
    $filename = 'invoice_' . $invoice['invoice_number'] . '.pdf';
    $filepath = '../uploads/invoices/' . $filename;
    
    // Create directory if it doesn't exist
    if (!is_dir('../uploads/invoices/')) {
        if (!mkdir('../uploads/invoices/', 0777, true)) {
            throw new Exception('Failed to create uploads directory');
        }
    }

    // Check if we can write to the directory
    if (!is_writable('../uploads/invoices/')) {
        throw new Exception('Uploads directory is not writable');
    }

    try {
        $pdf->Output($filepath, 'F');
    } catch (Exception $e) {
        throw new Exception('Failed to generate PDF file: ' . $e->getMessage());
    }

    // Check for any unexpected output
    $unexpected_output = ob_get_contents();
    if (!empty($unexpected_output)) {
        throw new Exception('Unexpected output detected: ' . substr($unexpected_output, 0, 200));
    }
    ob_end_clean();

    echo json_encode([
        'success' => true,
        'message' => 'PDF generated successfully',
        'filename' => $filename,
        'filepath' => $filepath,
        'download_url' => '../uploads/invoices/' . $filename
    ]);

} catch (Exception $e) {
    // Clean any output buffer
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    echo json_encode([
        'success' => false, 
        'message' => 'Error generating PDF: ' . $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ]);
}
?> 